#include "pch.h"
#include "Misc.hpp"
#include "LinksManager.h"
#include "LinksSAX.h"

// TODO: check existance of obligatory elements (e.g. defaultGroup)

/**********************
  CLinksContentHandler
 **********************/


CLinksContentHandler::CLinksContentHandler(CMiniSAXParser* aParser) {
	parser             = aParser;
	linkContentHandler = NULL;
	state              = ROOT_STATE;
}


CLinksContentHandler::~CLinksContentHandler() {
	if (linkContentHandler != NULL) {
		delete linkContentHandler;
	}
}


void CLinksContentHandler::startElement(const char* name, const CAttributeList& attributes)
        throw (CMiniSAXException)
{
	if (linkContentHandler != NULL) {
		linkContentHandler->startElement( name, attributes );
	}
	else {
		switch (state) {
		case ROOT_STATE:
			if (strcmp( name, "links" ) == 0) {
				state = LINKS_STATE;
				return;
			}
			break;
		case LINKS_STATE:
			if (strcmp( name, "link" ) == 0) {
				linkContentHandler = new CLinkContentHandler( parser );
				return;
			}
			break;
		}
		// if we are here then we have unexpected element
		throw CMiniSAXException( "Unexpected element", parser->getCurrentColumn(), parser->getCurrentRow() );
	}
}


void CLinksContentHandler::endElement(const char* name)
        throw (CMiniSAXException)
{
	if (linkContentHandler != NULL) {
		linkContentHandler->endElement( name );
		if (linkContentHandler->elementEnded()) {
			delete linkContentHandler;
			linkContentHandler = NULL;
		}
	}
	else {
		switch (state) {
		case LINKS_STATE:
			if (strcmp( name, "links" ) == 0) {
				state = ROOT_STATE;
				return;
			}
			break;
		}
		// if we are here then we have unexpected element
		throw CMiniSAXException( "Unexpected element", parser->getCurrentColumn(), parser->getCurrentRow() );
	}
}


void CLinksContentHandler::handleCharacters(const char* buffer, size_t len)
        throw (CMiniSAXException)
{
	if (linkContentHandler != NULL) {
		linkContentHandler->handleCharacters( buffer, len );
	}
	else {
		// ignore all characters
	}
}


/*********************
  CLinkContentHandler
 *********************/


CLinkContentHandler::CLinkContentHandler(CMiniSAXParser* aParser) {
	parser       = aParser;
	state		 = WAIT_STATE;
	link         = new CLink();
	endOfElement = false;
}


void CLinkContentHandler::startElement(const char* name, const CAttributeList& attributes)
	throw (CMiniSAXException)
{	
	// ensure that link memory will be freed
	try {

		if (endOfElement) {
			throw CMiniSAXException( "Illegal call: end of element already reached", 
									 parser->getCurrentColumn(), parser->getCurrentRow() );
		}

		switch (state) {
		case WAIT_STATE:
			if (strcmp( name, "name" ) == 0) {
				state = NAME_STATE;
				return;
			}
			if (strcmp( name, "address" ) == 0) {
				state = ADDRESS_STATE;
				return;
			}
			if (strcmp( name, "outbound" ) == 0) {
				// check type
				const CAttribute*	outboundType = attributes.getAttribute( "type" );
				if (outboundType != NULL) {
					if (strcmp( outboundType->getValue(), "binkley" ) == 0) {
						link->setOutboundType( OT_BINKLEY );
					}
					else 
					if (strcmp( outboundType->getValue(), "fileBoxes" ) == 0) {
						link->setOutboundType( OT_FILE_BOXES );
					}
					else {
						throw CMiniSAXException( "Unknown outbound type", 
												parser->getCurrentColumn(), parser->getCurrentRow() );
					}
				} else {
					// outbound type not specified, default is binkley
					link->setOutboundType( OT_BINKLEY );
				}

				const CAttribute*	flavour = attributes.getAttribute( "flavour" );
				if (flavour == NULL) {
					// default flavour is hold
					link->setFlavourType( FT_HOLD );
				} else {
					FlavourType	flavourType;
					if (strcmp( flavour->getValue(), "normal" ) == 0) {
						flavourType = FT_NORMAL;
					}
					else
					if (strcmp( flavour->getValue(), "direct" ) == 0) {
						flavourType = FT_DIRECT;
					}
					else
					if (strcmp( flavour->getValue(), "hold" ) == 0) {
						flavourType = FT_HOLD;
					}
					else
					if (strcmp( flavour->getValue(), "crash" ) == 0) {
						flavourType = FT_CRASH;
					}
					else
					if (strcmp( flavour->getValue(), "immediate" ) == 0) {
						flavourType = FT_IMMEDIATE;
					}
					else {
						throw CMiniSAXException( "Unknown outbound flavour", 
												 parser->getCurrentColumn(), parser->getCurrentRow() );
					}

					link->setFlavourType( flavourType );
				}

				// check file boxes flavour
				if (link->getOutboundType() == OT_FILE_BOXES) {
					if (link->getFlavourType() != FT_NORMAL && link->getFlavourType() != FT_HOLD) {
						throw CMiniSAXException( "FileBoxes outbound can have only either normal or hold flavour", 
												 parser->getCurrentColumn(), parser->getCurrentRow() );
					}
				}

				state = OUTBOUND_STATE;
				return;
			}
			if (strcmp( name, "allowAreaCreation" ) == 0) {
				state = ALLOW_AREA_CREATION_STATE;
				link->setAllowAreaCreation( true );
				return;
			}
			if (strcmp( name, "status" ) == 0) {
				state = STATUS_STATE;
				return;
			}
			if (strcmp( name, "robotName" ) == 0) {
				state = ROBOT_NAME_STATE;
				return;
			}
			if (strcmp( name, "availAreas" ) == 0) {
				state = AVAIL_AREAS_STATE;
				return;
			}
			if (strcmp( name, "useAKA" ) == 0) {
				state = USE_AKA_STATE;
				return;
			}
			if (strcmp( name, "security" ) == 0) {
				state = SECURITY_STATE;
				return;
			}
			break;
		case ALLOW_AREA_CREATION_STATE:
			if (strcmp( name, "path" ) == 0) {
				state = PATH_STATE;
				return;
			}
			if (strcmp( name, "defaultGroup" ) == 0) {
				state = DEFAULT_GROUP_STATE;
				return;
			}
			if (strcmp( name, "sendPriority" ) == 0) {
				state = AUTOCREATED_SEND_PRIORITY_STATE;
				return;
			}
			if (strcmp( name, "receivePriority" ) == 0) {
				state = AUTOCREATED_RECEIVE_PRIORITY_STATE;
				return;
			}
			break;
		case SECURITY_STATE:
			if (strcmp( name, "password" ) == 0) {
				state = PASSWORD_STATE;
				return;
			}
			if (strcmp( name, "sendPriority" ) == 0) {
				state = SEND_PRIORITY_STATE;
				return;
			}
			if (strcmp( name, "receivePriority" ) == 0) {
				state = RECEIVE_PRIORITY_STATE;
				return;
			}
			if (strcmp( name, "group" ) == 0) {
				state = GROUP_STATE;
				return;
			}
			break;
		}
		// if we are here then we failed to understand element
		throw CMiniSAXException( "Unknown element", 
								 parser->getCurrentColumn(), parser->getCurrentRow() );
	} catch (...) {
		if (link != NULL) {
			delete link;
			link = NULL;
		}
		throw;
	}
}


void CLinkContentHandler::endElement(const char* name)
    throw (CMiniSAXException)
{
	if (endOfElement) {
		throw CMiniSAXException( "Illegal call: end of element already reached", 
								 parser->getCurrentColumn(), parser->getCurrentRow() );
	}

	switch (state) {
	case WAIT_STATE:
		if (strcmp( name, "link" ) == 0) {
			endOfElement = true;
			if (!link->isAKASpecified()) {
				CLinksManager::chooseAKA( link );
			}
			CLinksManager::getInstance()->addLink( link );
			link = NULL;
			return;
		}
		break;
	case NAME_STATE:
		if (strcmp( name, "name" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case ADDRESS_STATE:
		if (strcmp( name, "address" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case OUTBOUND_STATE:
		if (strcmp( name, "outbound" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case STATUS_STATE:
		if (strcmp( name, "status" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case ROBOT_NAME_STATE:
		if (strcmp( name, "robotName" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case AVAIL_AREAS_STATE:
		if (strcmp( name, "availAreas" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case USE_AKA_STATE:
		if (strcmp( name, "useAKA" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case PATH_STATE:
		if (strcmp( name, "path" ) == 0) {
			state = ALLOW_AREA_CREATION_STATE;
			return;
		}
		break;
	case DEFAULT_GROUP_STATE:
		if (strcmp( name, "defaultGroup" ) == 0) {
			state = ALLOW_AREA_CREATION_STATE;
			return;
		}
		break;
	case AUTOCREATED_SEND_PRIORITY_STATE:
		if (strcmp( name, "sendPriority" ) == 0) {
			state = ALLOW_AREA_CREATION_STATE;
			return;
		}
		break;
	case AUTOCREATED_RECEIVE_PRIORITY_STATE:
		if (strcmp( name, "receivePriority" ) == 0) {
			state = ALLOW_AREA_CREATION_STATE;
			return;
		}
		break;
	case ALLOW_AREA_CREATION_STATE:
		if (strcmp( name, "allowAreaCreation" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	case PASSWORD_STATE:
		if (strcmp( name, "password" ) == 0) {
			state = SECURITY_STATE;
			return;
		}
		break;
	case SEND_PRIORITY_STATE:
		if (strcmp( name, "sendPriority" ) == 0) {
			state = SECURITY_STATE;
			return;
		}
		break;
	case RECEIVE_PRIORITY_STATE:
		if (strcmp( name, "receivePriority" ) == 0) {
			state = SECURITY_STATE;
			return;
		}
		break;
	case GROUP_STATE:
		if (strcmp( name, "group" ) == 0) {
			state = SECURITY_STATE;
			return;
		}
		break;
	case SECURITY_STATE:
		if (strcmp( name, "security" ) == 0) {
			state = WAIT_STATE;
			return;
		}
		break;
	}
	// if we are here then we failed to recognize element
	throw CMiniSAXException( "Unexpected end of element", 
							 parser->getCurrentColumn(), parser->getCurrentRow() );
}


void CLinkContentHandler::handleCharacters(const char* buffer, size_t len)
    throw (CMiniSAXException)
{
	if (endOfElement) {
		throw CMiniSAXException( "Illegal call: end of element already reached", 
								 parser->getCurrentColumn(), parser->getCurrentRow() );
	}

	switch (state) {
	case NAME_STATE:
		{
			string	name( buffer, len );
			LPTSTR	ansiName = new TCHAR[len + 1];
			::OemToChar( name.c_str(), ansiName );
			link->setName( ansiName);
			delete [] ansiName;
		}
		break;
	case ADDRESS_STATE:
		{
			tstring	addressString( buffer, len );
			SFTNAddress	address;
			if (CLinksManager::parseLinkAddress( addressString.c_str(), address )) {
				link->setAddress( address );
			} else {
				throw CMiniSAXException( "Invalid address", 
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
		}
		break;
	case STATUS_STATE:
		{
			tstring	status( buffer, len );
			int		activity = ACTIVITY_USUAL;
			
			if (strcmp( status.c_str(), "passive" ) == 0) {
				activity = ACTIVITY_PASSIVE;
			}
			else 
			if (strcmp( status.c_str(), "unavailable" ) == 0) {
				activity = ACTIVITY_UNAVAILABLE;
			}

			if (activity == ACTIVITY_USUAL) {
				throw CMiniSAXException( "Unknown activity", 
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}

			link->setActivity( activity );
		}
		break;
	case ROBOT_NAME_STATE:
		{
			string	name( buffer, len );
			LPTSTR	ansiName = new TCHAR[len + 1];
			::OemToChar( name.c_str(), ansiName );
			link->setRobotName( ansiName );
			delete [] ansiName;
		}
		break;
	case AVAIL_AREAS_STATE:
		{
			link->setAvailAreasFileName( tstring( buffer, len ) );
		}
		break;
	case USE_AKA_STATE:
		{
			tstring		akaValue( buffer, len );

			SFTNAddress	akaAddress;

			if (!CLinksManager::parseLinkAddress( akaValue.c_str(), akaAddress )) {
				throw CMiniSAXException( "Invalid AKA address", 
										 parser->getCurrentColumn(), parser->getCurrentRow() );			
			}

			// check AKA
            if (AddrCmp( g_vars.m_addrMain, akaAddress ) != 0) {
				int	i = 0;
				while (i < g_vars.m_vAKAAddr.size()) {
					if (AddrCmp( g_vars.m_vAKAAddr[i], akaAddress ) == 0) {
                        break;
					}
					++i;
                }
				if (i >= g_vars.m_vAKAAddr.size()) {
					// aka not found
					throw CMiniSAXException( "Unknown AKA address", 
											 parser->getCurrentColumn(), parser->getCurrentRow() );
				}
			}
			link->setOurAddress( akaAddress );
			link->setAKASpecified( true );
		}
		break;
	case PATH_STATE:
		{
			link->setAutoCreatedPathName( tstring( buffer, len ) );
		}
		break;
	case DEFAULT_GROUP_STATE:
		{
			link->setDefaultGroup( tstring( buffer, len ) );
		}
		break;
	case PASSWORD_STATE:
		{
			link->setPassword( tstring( buffer, len ) );
		}
		break;
	case SEND_PRIORITY_STATE:
		{
			tstring	priorityStr( buffer, len );
			int		priority;
			int		scanResult = sscanf( priorityStr.c_str(), "%d", &priority );
			if (scanResult == 0 || scanResult == EOF || priority < 0) {
				throw CMiniSAXException( "Invalid send priority",
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			link->setSendPriority( priority );
		}
		break;
	case RECEIVE_PRIORITY_STATE:
		{
			tstring	priorityStr( buffer, len );
			int		priority;
			int		scanResult = sscanf( priorityStr.c_str(), "%d", &priority );
			if (scanResult == 0 || scanResult == EOF || priority < 0) {
				throw CMiniSAXException( "Invalid receive priority",
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			link->setReceivePriority( priority );
		}
		break;
	case AUTOCREATED_SEND_PRIORITY_STATE:
		{
			tstring	priorityStr( buffer, len );
			int		priority;
			int		scanResult = sscanf( priorityStr.c_str(), "%d", &priority );
			if (scanResult == 0 || scanResult == EOF || priority < 0) {
				throw CMiniSAXException( "Invalid autocreated send priority",
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			link->setAutocreatedSendPriority( priority );
		}
		break;
	case AUTOCREATED_RECEIVE_PRIORITY_STATE:
		{
			tstring	priorityStr( buffer, len );
			int		priority;
			int		scanResult = sscanf( priorityStr.c_str(), "%d", &priority );
			if (scanResult == 0 || scanResult == EOF || priority < 0) {
				throw CMiniSAXException( "Invalid autocreated receive priority",
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			link->setAutocreatedReceivePriority( priority );
		}
		break;
	case GROUP_STATE:
		{
			tstring	group( buffer, len );
			if (link->canAccessGroup( group )) {
				throw CMiniSAXException( "Duplicate group",
										 parser->getCurrentColumn(), parser->getCurrentRow() );
			}
			link->addGroup( group );
		}
		break;
	}
}
